/*
 * ebt_netmap
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "../include/ebtables_u.h"
#include <netinet/ether.h>
#include <ebt_netmap.h>

enum {
	NETMAP_SRC_FROM = '1',
	NETMAP_SRC_TO,
	NETMAP_DST_FROM,
	NETMAP_DST_TO,
};

static struct option opts[] =
{
	{ "netmap-src-from" ,  required_argument, 0, NETMAP_SRC_FROM },
	{ "netmap-src-to" ,    required_argument, 0, NETMAP_SRC_TO },
	{ "netmap-dst-from" ,  required_argument, 0, NETMAP_DST_FROM },
	{ "netmap-dst-to" ,    required_argument, 0, NETMAP_DST_TO },
	{ 0 }
};

static void print_help()
{
	printf(
	"netmap target options:\n"
	" --netmap-src-from      : src from base ip address\n"
	" --netmap-src-to        : src to base ip address\n"
	" --netmap-dst-from      : dst from base ip address\n"
	" --netmap-dst-to        : dst to base ip address\n");
}

static void init(struct ebt_entry_target *target)
{
	struct ebt_netmap_info *info = (void *)target->data;
	memset(info, 0, sizeof(*info));
}

static int parse(int c, char **argv, int argc,
   const struct ebt_u_entry *entry, unsigned int *flags,
   struct ebt_entry_target **target)
{
	struct ebt_netmap_info *info = (void *)(*target)->data;
	uint32_t *ip = NULL;
	uint32_t mask;
	const char *msg;
	switch (c) {
	case NETMAP_SRC_FROM:
		ip = &info->src.from;
		msg = "src-from";
		info->bitmask |= EBT_NETMAP_SRC;
		break;
	case NETMAP_SRC_TO:
		ip = &info->src.to;
		msg = "src-to";
		info->bitmask |= EBT_NETMAP_SRC;
		break;
	case NETMAP_DST_FROM:
		ip = &info->dst.from;
		msg = "dst-from";
		info->bitmask |= EBT_NETMAP_DST;
		break;
	case NETMAP_DST_TO:
		ip = &info->dst.to;
		msg = "dst-to";
		info->bitmask |= EBT_NETMAP_DST;
		break;
	default:
		return 0;
	}
	ebt_check_option2(flags, 1 << (c-NETMAP_SRC_FROM));
	ebt_parse_ip_address(optarg, ip, &mask);
	if (ebt_errormsg[0] || mask != ~0) {
		ebt_print_error2("Problem with specified %s", msg);
	}
	return 1;
}

static void final_check(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target, const char *name,
   unsigned int hookmask, unsigned int time)
{
	struct ebt_netmap_info *info = (void *)target->data;
	if ((entry->ethproto != ETH_P_IP && entry->ethproto != ETH_P_ARP)
	||  entry->invflags & EBT_IPROTO) {
		ebt_print_error("The protocol must be specified as IP or ARP");
	} else if (!info->bitmask) {
		ebt_print_error("No mapping specified");
	} else if (strcmp(name, "nat")) {
		ebt_print_error("netmap only allowed in nat table");
	} else {
		CLEAR_BASE_CHAIN_BIT;
		if (hookmask & ~((1 << NF_BR_PRE_ROUTING) | (1 << NF_BR_POST_ROUTING))) {
			ebt_print_error("netmap only allowed in PREROUTING or POSTROUTING");
		}
	}
}

static void print_ip(void *ip)
{
	unsigned char *p = ip;
	int i;
	for (i=0; i < 4; ++i, ++p)
		printf("%s%d", i ? "." : "", *p);
}

static void print(const struct ebt_u_entry *entry,
   const struct ebt_entry_target *target)
{
	struct ebt_netmap_info *info = (void *)target->data;
	if (info->bitmask & EBT_NETMAP_SRC) {
		printf(" --netmap-src-from "); print_ip(&info->src.from);
		printf(" --netmap-src-to "); print_ip(&info->src.to);
	}
	if (info->bitmask & EBT_NETMAP_DST) {
		printf(" --netmap-dst-from "); print_ip(&info->dst.from);
		printf(" --netmap-dst-to "); print_ip(&info->dst.to);
	}
}

static int compare(const struct ebt_entry_target *t1,
   const struct ebt_entry_target *t2)
{
	struct ebt_netmap_info *info1 = (void *)t1->data;
	struct ebt_netmap_info *info2 = (void *)t2->data;
	if (info1->bitmask != info2->bitmask) return 0;
	if (info1->bitmask & EBT_NETMAP_SRC) {
		if (info1->src.from != info2->src.from) return 0;
		if (info1->src.to != info2->src.to) return 0;
	}
	if (info1->bitmask & EBT_NETMAP_DST) {
		if (info1->dst.from != info2->dst.from) return 0;
		if (info1->dst.to != info2->dst.to) return 0;
	}
	return 1;
}

static struct ebt_u_target netmap_target =
{
	.name		= "netmap",
	.size		= sizeof(struct ebt_netmap_info),
	.help		= print_help,
	.init		= init,
	.parse		= parse,
	.final_check= final_check,
	.print		= print,
	.compare	= compare,
	.extra_ops	= opts,
};

void _init(void)
{
	ebt_register_target(&netmap_target);
}
