/*
 *  ebt_netmap
 */
#include <linux/ip.h>
#include <net/ip.h>
#include <linux/if_arp.h>
#include <net/arp.h>
#include <linux/in.h>
#include <linux/module.h>
#include <linux/netfilter/x_tables.h>
#include <linux/netfilter_bridge/ebtables.h>
#include <linux/netfilter_bridge/ebt_nat.h>
#include "ebt_netmap.h"

#if 0
#define PR_DEBUG(x...) printk(x)
#else
#define PR_DEBUG(x...)
#endif

static unsigned ebt_netmap_ip(__be32 *addr, const struct ebt_netmap_map *map)
{
	unsigned ip = ntohl(*addr);
	ip -= ntohl(map->from);
	ip += ntohl(map->to);
	ip = htonl(ip);
	PR_DEBUG("ebtables netmap %pI4 -> %pI4\n", addr, &ip);
	*addr = ip;
	return 1;
}

static unsigned int
ebt_netmap_tg_ip(struct sk_buff *skb, const struct xt_action_param *par)
{
	const struct ebt_netmap_info *info = par->targinfo;
	struct iphdr *ih;
	struct udphdr *uh;
	if (skb_headlen(skb) < sizeof(*ih)) return EBT_CONTINUE;
	ih = (void*)(skb->data + 0);
	/* only udp for now (check sum fixup) */
	if (ih->protocol != IPPROTO_UDP) return EBT_CONTINUE;
	if (skb_headlen(skb) < ih->ihl*4 + sizeof(*uh)) return EBT_CONTINUE;
	if (!skb_make_writable(skb, 0)) {
		printk("ebtable netmap ip: can not write to skb\n");
		return EBT_CONTINUE;
	}
	if (info->bitmask & EBT_NETMAP_SRC) ebt_netmap_ip(&ih->saddr, &info->src);
	if (info->bitmask & EBT_NETMAP_DST) ebt_netmap_ip(&ih->daddr, &info->dst);
	ih->check = 0;
	ih->check = ip_fast_csum((unsigned char *)ih, ih->ihl);
	uh = (void*)(skb->data + ih->ihl*4);
	uh->check = 0;
	PR_DEBUG("ebtable netmap: ip message fixed src %pI4 dst %pI4\n", &ih->saddr, &ih->daddr);
	return EBT_CONTINUE;
}

static unsigned int
ebt_netmap_tg_arp(struct sk_buff *skb, const struct xt_action_param *par)
{
	const struct ebt_netmap_info *info = par->targinfo;
	struct arphdr *ah;
	__be32 *sip;
	__be32 *dip;
	/* arphdr shw[hwlen] sip[iplen] thw[hwlen] tip[iplen] */
	if (skb_headlen(skb) < sizeof(*ah)+2*ETH_ALEN+2*4) return EBT_CONTINUE;
	ah = (void*)(skb->data + 0);
	if (ntohs(ah->ar_op) != ARPOP_REQUEST && ntohs(ah->ar_op) != ARPOP_REPLY)
		return EBT_CONTINUE;
	if (ah->ar_hln != ETH_ALEN) return EBT_CONTINUE;
	if (ah->ar_pro != htons(ETH_P_IP)) return EBT_CONTINUE;
	if (ah->ar_pln != 4) return EBT_CONTINUE;
	if (!skb_make_writable(skb, 0)) {
		printk("ebtable netmap arp: can not write to skb\n");
		return EBT_CONTINUE;
	}
	sip = (void *)(skb->data + sizeof(*ah)) + ETH_ALEN;
	dip = (void *)(skb->data + sizeof(*ah)) + ETH_ALEN*2 + 4;
	if (info->bitmask & EBT_NETMAP_SRC) ebt_netmap_ip(sip, &info->src);
	if (info->bitmask & EBT_NETMAP_DST) ebt_netmap_ip(dip, &info->dst);
	PR_DEBUG("ebtable netmap: arp message %d fixed src %pI4 dst %pI4\n",
		ntohs(ah->ar_op), sip, dip);
	return EBT_CONTINUE;
}

static unsigned int
ebt_netmap_tg(struct sk_buff *skb, const struct xt_action_param *par)
{
	if (eth_hdr(skb)->h_proto == htons(ETH_P_IP))
		return ebt_netmap_tg_ip(skb, par);
	if (eth_hdr(skb)->h_proto == htons(ETH_P_ARP))
		return ebt_netmap_tg_arp(skb, par);
	PR_DEBUG("ebtable netmap: not arp or ip 0x%x", ntohs(eth_hdr(skb)->h_proto));
	return EBT_CONTINUE;
}

static int ebt_netmap_tg_check(const struct xt_tgchk_param *par)
{
	const struct ebt_netmap_info *info = par->targinfo;
	const struct ebt_entry *e = par->entryinfo;
	if (e->invflags & EBT_IPROTO) return -EINVAL;
	if (e->ethproto != htons(ETH_P_IP) && e->ethproto != htons(ETH_P_ARP))
		return -EINVAL;
	if (info->bitmask == 0 || info->bitmask & ~(EBT_NETMAP_SRC|EBT_NETMAP_DST))
		return -EINVAL;
	PR_DEBUG("netmap passed check mask %x src %pI4 -> %pI4 dst %pI4 -> %pI4\n",
		info->bitmask,
		&info->src.from, &info->src.to,
		&info->dst.from, &info->dst.to);
	return 0;
}

static struct xt_target ebt_netmap_tg_reg __read_mostly = {
	.name		= EBT_NETMAP_TARGET,
	.revision	= 0,
	.family		= NFPROTO_BRIDGE,
	.table		= "nat",
	.hooks		= (1 << NF_BR_NUMHOOKS) | (1 << NF_BR_PRE_ROUTING) | (1 << NF_BR_POST_ROUTING),
	.target		= ebt_netmap_tg,
	.checkentry	= ebt_netmap_tg_check,
	.targetsize	= sizeof(struct ebt_netmap_info),
	.me		= THIS_MODULE,
};

static int __init ebt_netmap_init(void)
{
	return xt_register_target(&ebt_netmap_tg_reg);
}

static void __exit ebt_netmap_fini(void)
{
	xt_unregister_target(&ebt_netmap_tg_reg);
}

module_init(ebt_netmap_init);
module_exit(ebt_netmap_fini);
MODULE_DESCRIPTION("Ebtables: Netmap");
MODULE_LICENSE("GPL");
