-- Copyright 2008 Steven Barth <steven@midlink.org>
-- Copyright 2008 Jo-Philipp Wich <jow@openwrt.org>
-- Copyright 2017 Eric Luehrsen <ericluehrsen@hotmail.com>
-- Licensed to the public under the Apache License 2.0.

module("luci.controller.unbound", package.seeall)


function index()
  local ucl = luci.model.uci.cursor()
  local valexp = ucl:get_first("unbound", "unbound", "extended_luci")
  local valman = ucl:get_first("unbound", "unbound", "manual_conf")


  if not nixio.fs.access("/etc/config/unbound") then
    return
  end


  if valexp == "1" then
    -- Expanded View
    entry({"admin", "services", "unbound"}, firstchild(), _("Recursive DNS")).dependent = false

    -- UCI Tab(s)
    entry({"admin", "services", "unbound", "configure"}, cbi("unbound/configure"), _("Settings"), 10)

    -- Status Tab(s)
    entry({"admin", "services", "unbound", "status"}, firstchild(), _("Status"), 20)
    entry({"admin", "services", "unbound", "status", "syslog"}, call("QuerySysLog"), _("Log"), 50).leaf = true


    if nixio.fs.access("/usr/sbin/unbound-control") then
      -- Require unbound-control to execute
      entry({"admin", "services", "unbound", "status", "statistics"}, call("QueryStatistics"), _("Statistics"), 10).leaf = true
      entry({"admin", "services", "unbound", "status", "localdata"}, call("QueryLocalData"), _("Local Data"), 20).leaf = true
      entry({"admin", "services", "unbound", "status", "localzone"}, call("QueryLocalZone"), _("Local Zones"), 30).leaf = true

    else
      entry({"admin", "services", "unbound", "status", "statistics"}, call("ShowEmpty"), _("Statistics"), 10).leaf = true
    end


    -- Raw File Tab(s)
    entry({"admin", "services", "unbound", "files"}, firstchild(), _("Files"), 30)


    if valman ~= "1" then
      entry({"admin", "services", "unbound", "files", "base"}, call("ShowUnboundConf"), _("UCI: Unbound"), 10).leaf = true
    else
      entry({"admin", "services", "unbound", "files", "base"}, form("unbound/manual"), _("Edit: Unbound"), 10).leaf = true
    end


    entry({"admin", "services", "unbound", "files", "server"}, form("unbound/server"), _("Edit: Server"), 20).leaf = true
    entry({"admin", "services", "unbound", "files", "extended"}, form("unbound/extended"), _("Edit: Extended"), 30).leaf = true


    if nixio.fs.access("/var/lib/unbound/unbound_dhcp.conf") then
      entry({"admin", "services", "unbound", "files", "dhcp"}, call("ShowDHCPConf"), _("Include: DHCP"), 40).leaf = true
    end


    if nixio.fs.access("/var/lib/unbound/adb_list.overall") then
      entry({"admin", "services", "unbound", "files", "adblock"}, call("ShowAdblock"), _("Include: Adblock"), 50).leaf = true
    end

  else
    -- Simple View to UCI only
    entry({"admin", "services", "unbound"}, cbi("unbound/configure"), _("Recursive DNS")).dependent = false
  end
end


function ShowEmpty()
  local lclhead = "Unbound Control"
  local lcldesc = luci.i18n.translate("This could display more statistics with the unbound-control package.")
  luci.template.render("unbound/show-empty", {heading = lclhead, description = lcldesc})
end


function QuerySysLog()
  local lclhead = "System Log"
  local lcldata = luci.util.exec("logread | grep -i unbound")
  local lcldesc = luci.i18n.translate("This shows syslog filtered for events involving Unbound.")
  luci.template.render("unbound/show-textbox", {heading = lclhead, description = lcldesc, content = lcldata})
end


function QueryStatistics()
  local lclhead = "Unbound Control Stats"
  local lcldata = luci.util.exec("unbound-control -c /var/lib/unbound/unbound.conf stats_noreset")
  local lcldesc = luci.i18n.translate("This shows some performance statistics tracked by Unbound.")
  luci.template.render("unbound/show-textbox", {heading = lclhead, description = lcldesc, content = lcldata})
end


function QueryLocalData()
  local lclhead = "Unbound Control Local Data"
  local lcldata = luci.util.exec("unbound-control -c /var/lib/unbound/unbound.conf list_local_data")
  local lcldesc = luci.i18n.translate("This shows local host records that shortcut recursion.")
  luci.template.render("unbound/show-textbox", {heading = lclhead, description = lcldesc, content = lcldata})
end


function QueryLocalZone()
  local lclhead = "Unbound Control Local Zones"
  local lcldata = luci.util.exec("unbound-control -c /var/lib/unbound/unbound.conf list_local_zones")
  local lcldesc = luci.i18n.translate("This shows local zone definitions that affect recursion routing or processing. ")
  luci.template.render("unbound/show-textbox", {heading = lclhead, description = lcldesc, content = lcldata})
end


function ShowUnboundConf()
  local unboundfile = "/var/lib/unbound/unbound.conf"
  local lclhead = "Unbound Conf"
  local lcldata = nixio.fs.readfile(unboundfile)
  local lcldesc = luci.i18n.translate("This shows configuration generated by UCI:")
  lcldesc = lcldesc .. " (" .. unboundfile .. ")"
  luci.template.render("unbound/show-textbox", {heading = lclhead, description = lcldesc, content = lcldata})
end


function ShowDHCPConf()
  local dhcpfile = "/var/lib/unbound/unbound_dhcp.conf"
  local lclhead = "DHCP Conf"
  local lcldata = nixio.fs.readfile(dhcpfile)
  local lcldesc = luci.i18n.translate("This shows LAN hosts added by DHCP hook scripts:")
  lcldesc = lcldesc .. " (" .. dhcpfile .. ")"
  luci.template.render("unbound/show-textbox", {heading = lclhead, description = lcldesc, content = lcldata})
end


function ShowAdblock()
  local adblockfile = "/var/lib/unbound/adb_list.overall"
  local lclhead = "Adblock Conf"
  local lcldata, lcldesc


  if nixio.fs.stat(adblockfile).size > 262144 then
    lcldesc = luci.i18n.translate("Adblock domain list is too large for LuCI:")
    lcldesc = lcldesc .. " (" .. adblockfile .. ")"
    luci.template.render("unbound/show-empty", {heading = lclhead, description = lcldesc})

  else
    lcldata = nixio.fs.readfile(adblockfile)
    lcldesc = luci.i18n.translate("This shows blocked domains provided by Adblock scripts:")
    lcldesc = lcldesc .. " (" .. adblockfile .. ")"
    luci.template.render("unbound/show-textbox", {heading = lclhead, description = lcldesc, content = lcldata})
  end
end

